  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief This file includes the DH ATP tests
   *
   *  \version CRYS_SEP_ATP_DH_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern CRYSError_t      TST_ATP_Error;
/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   DxUint8_t   *FailureFunctionNameOut_ptr);

/* global variables could occupied only 1K,   */
/* so all variables are allocated as internal */
/* variables of the functions                 */ 



/***********************************************************************
* Function Name:                                                           
*  SEPQA_DH_PKCS3         
* 
* Inputs:
* Generator_ptr 		- Pointer to the Generator octet string
* GeneratorSize 		- Size of the Generator String (in bytes)
* Prime_ptr 			- Pointer to the Prime octet string
* PrimeSize 			- Size of the Prime string (in bytes)
* L 					- Relevant only for PKCS#3, the private value length in bits
*	If L != 0 then - force the private key to be 2^(L-1) <= Prv < 2^l
*	if L 0 then: 0< Prv < P-1
*	In ANSI X9.42 L is irrelevant  
* FailureFunctionNameOut - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes the Diffie-Hellman key agreement 
*  scheme according to PKCS#3  
* Algorithm:   
* 1. Call the CRYS_DH_PKCS3_GeneratePubPrv function to 
*    generate a public and privet key for the client
* 2. Call the CRYS_DH_PKCS3_GeneratePubPrv function to 
*    generate a public and privet key for the Server
* 3. Call the CRYS_DH_PKCS3_GetSecretKey function to 
*    generate a secret key for the client  
* 4. Call the CRYS_DH_PKCS3_GetSecretKey function to 
*    generate a secret key for the Server
* 5. Compare the two secret keys - to verify they are identical                                                    
*
**************************************************************************************/                                
CRYSError_t SEPQA_DH_PKCS3(DxUint8_t *Generator_ptr,
						   DxUint16_t GeneratorSize,
						   DxUint8_t *Prime_ptr,
						   DxUint16_t PrimeSize,
						   DxUint16_t L,
						   DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t			  TST_TestStatus = 0;

	CRYS_RSAUserPubKey_t	*TST_UserPubKey; 
	CRYS_RSAPrimeData_t		*TST_PrimeData;
	DxUint8_t               *TST_ClientPrvKey;	
	DxUint16_t              TST_ClientPrvKeySize;
	DxUint8_t               *TST_ClientPubKey;
	DxUint16_t              TST_ClientPubKeySize;
	DxUint8_t               *TST_ServerPrvKey;
	DxUint16_t              TST_ServerPrvKeySize;
	DxUint8_t               *TST_ServerPubKey;
	DxUint16_t              TST_ServerPubKeySize;   
	DxUint8_t               *TST_ClientSecretKey;
	DxUint16_t              TST_ClientSecretKeySize;
	DxUint8_t               *TST_ServerSecretKey;
	DxUint16_t              TST_ServerSecretKeySize;  

	/* 1. Allocate memory buffers */
	/******************************/
	TST_UserPubKey		 = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_PrimeData        = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));
	TST_ClientPrvKey     = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ClientPubKey     = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerPrvKey	 = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerPubKey	 = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ClientSecretKey  = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerSecretKey  = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);

	if ((TST_UserPubKey == DX_NULL)     ||
		(TST_PrimeData == DX_NULL)      ||
		(TST_ClientPrvKey == DX_NULL)      ||
		(TST_ClientPubKey == DX_NULL)      ||
		(TST_ServerPrvKey == DX_NULL)      ||
		(TST_ServerPubKey == DX_NULL)      ||
		(TST_ClientSecretKey == DX_NULL)             ||
		(TST_ServerSecretKey == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut);

		goto End;                                                                       
	}

	/* 1. Generating Keys for the Client side */
	/******************************************/   
	TST_ClientPrvKeySize = TST_ClientPubKeySize = TST_ServerPrvKeySize = TST_ServerPubKeySize =  PrimeSize;

	TST_ATP_Error =  CRYS_DH_PKCS3_GeneratePubPrv(Generator_ptr,
												GeneratorSize,
												Prime_ptr,
												PrimeSize,
												L,
												TST_UserPubKey,
												TST_PrimeData,
												TST_ClientPrvKey,
												&TST_ClientPrvKeySize,
												TST_ClientPubKey,
												&TST_ClientPubKeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_DH_PKCS3_GeneratePubPrv",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End; 

	/* 2. Generating Keys for the Server side */
	/******************************************/   
	TST_ATP_Error =  CRYS_DH_PKCS3_GeneratePubPrv(Generator_ptr,
		GeneratorSize,
		Prime_ptr,
		PrimeSize,
		L,
		TST_UserPubKey,
		TST_PrimeData,
		TST_ServerPrvKey,
		&TST_ServerPrvKeySize,
		TST_ServerPubKey,
		&TST_ServerPubKeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_DH_PKCS3_GeneratePubPrv",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End; 


	/* 3. Generating Secret Key for the client side */ 
	/************************************************/ 
	TST_ServerSecretKeySize = TST_ClientSecretKeySize = PrimeSize;

	TST_ATP_Error =  CRYS_DH_PKCS3_GetSecretKey(TST_ClientPrvKey,
		TST_ClientPrvKeySize,
		TST_ServerPubKey,
		TST_ServerPubKeySize,                                   
		Prime_ptr,
		PrimeSize,
		TST_UserPubKey,
		TST_PrimeData,                                     
		TST_ClientSecretKey,
		&TST_ClientSecretKeySize);      

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_DH_PKCS3_GetSecretKey",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End; 

	/* 4. Generating Secret Key for the server side */ 
	/************************************************/ 
	TST_ATP_Error =  CRYS_DH_PKCS3_GetSecretKey(TST_ServerPrvKey,
		TST_ServerPrvKeySize,
		TST_ClientPubKey,
		TST_ClientPubKeySize,                                   
		Prime_ptr,
		PrimeSize,
		TST_UserPubKey,
		TST_PrimeData,                                     
		TST_ServerSecretKey,
		&TST_ServerSecretKeySize);      

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_DH_PKCS3_GetSecretKey",
		FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End; 

	/* 5. Verifying we have the same Secret Key both for client & Server */
	/*********************************************************************/                                                    
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_ServerSecretKey ,
		TST_ClientSecretKey,
		TST_ServerSecretKeySize); 


	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut);

End:

	DX_VOS_MemFree(TST_UserPubKey);
	DX_VOS_MemFree(TST_PrimeData);
	DX_VOS_MemFree(TST_ClientPrvKey);
	DX_VOS_MemFree(TST_ClientPubKey);
	DX_VOS_MemFree(TST_ServerPrvKey);
	DX_VOS_MemFree(TST_ServerPubKey);
	DX_VOS_MemFree(TST_ClientSecretKey);
	DX_VOS_MemFree(TST_ServerSecretKey);

	return TST_TestStatus;	

}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_DH_ANSI942         
* 
* Inputs:
* Generator_ptr 		- Pointer to the Generator octet string
* GeneratorSize 		- Size of the Generator String (in bytes)
* Prime_ptr 			- Pointer to the Prime octet string
* PrimeSize 			- Size of the Prime string (in bytes)
* Q_ptr 				- Relevant only for ANSI X9.42 - Pointer to the Q octet string
*							1<= Prv <= q-1   or   1< Prv < q-1
* QSize	 			    - Relevant only for ANSI X9.42 - Size of the Q string (in bytes)  				 				  
* SecretDataGenType     - secret data gen type
* KeyGetSecretKeyType   - get secret key type
* KeyLenInBits          - key len in bits
* FailureFunctionNameOut - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes the Diffie-Hellman key agreement 
*  scheme according to PKCS#3  
* Algorithm:   
* 1. Call the CRYS_DH_ANSI_X942_GeneratePubPrv function to 
*    generate a public and privet key for the client
* 2. Call the CRYS_DH_ANSI_X942_GeneratePubPrv function to 
*    generate a public and privet key for the Server
* 3. Repete steps 1 & 2 in case of Hybrid scheam
* 4. Check the server and client public keys according to their domain parameters
* 5. Call the CRYS_DH_X942_GetSecret_ASN1_2_Data
*             or 
* 	       CRYS_DH_X942_GetSecret_Concat_2_Data
*             and in case of Hybrid scheam
*             CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data
*             or 
*             CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data
*     both for the server and the Client to generate the secret keys
* 6. Compare the two secret keys - to verify they are identical													
*
**************************************************************************************/                                
CRYSError_t SEPQA_DH_ANSI942(DxUint8_t *Generator_ptr,
						   DxUint16_t   GeneratorSize,
						   DxUint8_t   *Prime_ptr,
						   DxUint16_t   PrimeSize,
						   DxUint8_t   *Q_ptr,
						   DxUint16_t   QSize,
						   DxUint16_t   SecretDataGenType,
						   DxUint16_t   KeyGetSecretKeyType,
						   DxUint16_t	KeyLenInBits,
						   DxUint8_t   *FailureFunctionNameOut)
{
	/*** TST Variables Definitions ****/
	CRYSError_t			  TST_TestStatus = 0;

	CRYS_DHKG_ExpTemp_t	    *TST_TempBuff;
	CRYS_RSAUserPubKey_t	*TST_UserPubKey; 
	CRYS_RSAPrimeData_t		*TST_PrimeData;
	DxUint8_t               *TST_ClientPrvKey1;	
	DxUint16_t              TST_ClientPrvKeySize1;
	DxUint8_t               *TST_ClientPrvKey2;	
	DxUint16_t              TST_ClientPrvKeySize2;
	DxUint8_t               *TST_ClientPubKey1;
	DxUint16_t              TST_ClientPubKeySize1;
	DxUint8_t               *TST_ClientPubKey2;
	DxUint16_t              TST_ClientPubKeySize2;
	DxUint8_t               *TST_ServerPrvKey1;
	DxUint16_t              TST_ServerPrvKeySize1;
	DxUint8_t               *TST_ServerPrvKey2;
	DxUint16_t              TST_ServerPrvKeySize2;
	DxUint8_t               *TST_ServerPubKey1;
	DxUint16_t              TST_ServerPubKeySize1;   
	DxUint8_t               *TST_ServerPubKey2;
	DxUint16_t              TST_ServerPubKeySize2;   
	DxUint8_t               *TST_ClientSecretKey;
	DxUint16_t              TST_ClientSecretKeySize;
	DxUint8_t               *TST_ServerSecretKey;
	DxUint16_t              TST_ServerSecretKeySize;  


	/* 1. Allocate memory buffers */
	/******************************/
	TST_TempBuff         = DX_VOS_MemMalloc(sizeof(CRYS_DHKG_ExpTemp_t));
	TST_UserPubKey		 = DX_VOS_MemMalloc(sizeof(CRYS_RSAUserPubKey_t));
	TST_PrimeData        = DX_VOS_MemMalloc(sizeof(CRYS_RSAPrimeData_t));
	TST_ClientPrvKey1     = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ClientPubKey1     = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ClientPrvKey2     = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ClientPubKey2     = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerPrvKey1	 = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerPubKey1	 = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerPrvKey2	 = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerPubKey2	 = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ClientSecretKey  = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);
	TST_ServerSecretKey  = DX_VOS_MemMalloc(TST_MAX_DH_KEY_SIZE);

	if ((TST_TempBuff == DX_NULL)       ||
		(TST_UserPubKey == DX_NULL)     ||
		(TST_PrimeData == DX_NULL)      ||
		(TST_ClientPrvKey1 == DX_NULL)   ||
		(TST_ClientPubKey1 == DX_NULL)   ||
		(TST_ClientPrvKey2 == DX_NULL)   ||
		(TST_ClientPubKey2 == DX_NULL)   ||
		(TST_ServerPrvKey1 == DX_NULL)   ||
		(TST_ServerPubKey1 == DX_NULL)   ||
		(TST_ServerPrvKey2 == DX_NULL)   ||
		(TST_ServerPubKey2 == DX_NULL)   ||
		(TST_ClientSecretKey == DX_NULL)||
		(TST_ServerSecretKey == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut);

		goto End;                                                                       
	}

	/* 1. Generating Keys for the Client side */
	/******************************************/  
	TST_ClientPrvKeySize1 = TST_ClientPubKeySize1 = 
	TST_ClientPrvKeySize2 = TST_ClientPubKeySize2 =
	TST_ServerPrvKeySize1 = TST_ServerPubKeySize1 =  
	TST_ServerPrvKeySize2 = TST_ServerPubKeySize2 = PrimeSize;

	TST_ATP_Error = CRYS_DH_ANSI_X942_GeneratePubPrv(Generator_ptr, 
													 GeneratorSize, 
													 Prime_ptr, 
													 PrimeSize, 
													 Q_ptr, 
													 QSize, 
													 TST_UserPubKey,
													 TST_PrimeData,
													 TST_ClientPrvKey1,
													 &TST_ClientPrvKeySize1,
													 TST_ClientPubKey1,
													 &TST_ClientPubKeySize1); 

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_DH_ANSI_X942_GeneratePubPrv",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End; 

	/* 2. Generating Keys for the Server side */
	/******************************************/   
	TST_ATP_Error = CRYS_DH_ANSI_X942_GeneratePubPrv(Generator_ptr, 
													 GeneratorSize, 
													 Prime_ptr, 
													 PrimeSize, 
													 Q_ptr, 
													 QSize, 
													 TST_UserPubKey,
													 TST_PrimeData,
													 TST_ServerPrvKey1,
													 &TST_ServerPrvKeySize1,
													 TST_ServerPubKey1,
													 &TST_ServerPubKeySize1); 

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_DH_ANSI_X942_GeneratePubPrv",
												FailureFunctionNameOut);

	if (TST_TestStatus != TESTPASS) 
		goto End;     

	/* 3. Generating another Pair of Secret Keys both for Client and Serve */
	/* In case of working with Hybrid mode                              */
	/********************************************************************/
	if (SecretDataGenType == TST_DH_HYBRID_MODE)
	{
		/* Generating Keys #2 for the Client side */
		/******************************************/    
		TST_ATP_Error = CRYS_DH_ANSI_X942_GeneratePubPrv(
			Generator_ptr, 
			GeneratorSize, 
			Prime_ptr, 
			PrimeSize, 
			Q_ptr, 
			QSize, 
			TST_UserPubKey,
			TST_PrimeData,
			TST_ClientPrvKey2,
			&TST_ClientPrvKeySize2,
			TST_ClientPubKey2,
			&TST_ClientPubKeySize2); 

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
						"CRYS_DH_ANSI_X942_GeneratePubPrv - Hybrid #2 - Client",
						FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End; 

		/* 3. Generating Keys #2 for the Server side */
		/******************************************/    
		TST_ATP_Error = CRYS_DH_ANSI_X942_GeneratePubPrv(
			Generator_ptr,
			GeneratorSize,
			Prime_ptr, 
			PrimeSize, 
			Q_ptr, 
			QSize, 			
			TST_UserPubKey,
			TST_PrimeData,
			TST_ServerPrvKey2,
			&TST_ServerPrvKeySize2,
			TST_ServerPubKey2,
			&TST_ServerPubKeySize2);

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
			TST_ATP_Error,
			"CRYS_DH_ANSI_X942_GeneratePubPrv - Hybrid #2 - Server",
			FailureFunctionNameOut);

		if (TST_TestStatus != TESTPASS) 
			goto End;           
	}

	TST_ClientSecretKeySize = TST_ServerSecretKeySize = KeyLenInBits;

	if (SecretDataGenType == TST_DH_NON_HYBRID_MODE)
	{
		if (KeyGetSecretKeyType == TST_DH_GET_SECRET_KEY_ASN1)
		{
			/* 5. Generating Secret key for the Client */
			/*******************************************/
			TST_ATP_Error = CRYS_DH_X942_GetSecret_ASN1_2_Data(
				TST_UserPubKey, 
				TST_PrimeData,
				CRYS_DH_HASH_SHA1_mode,
				TST_ClientPrvKey1,
				TST_ClientPrvKeySize1, 
				TST_ServerPubKey1,
				TST_ServerPubKeySize1, 
				Prime_ptr,
				PrimeSize,
				TST_ClientSecretKey, 
				TST_ClientSecretKeySize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_GetSecret_ASN1_2_Data - Client",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 

			/* 5. Generating Secret key for the Server */
			/*******************************************/
			TST_ATP_Error = CRYS_DH_X942_GetSecret_ASN1_2_Data(
				TST_UserPubKey, 
				TST_PrimeData,
				CRYS_DH_HASH_SHA1_mode,
				TST_ServerPrvKey1,
				TST_ServerPrvKeySize1, 
				TST_ClientPubKey1,
				TST_ClientPubKeySize1, 
				Prime_ptr,
				PrimeSize,
				TST_ServerSecretKey, 
				TST_ServerSecretKeySize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_GetSecret_ASN1_2_Data - Server",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 
		}
		else
		{
			/* Generating Secret key for the Client */
			/****************************************/                  
			TST_ATP_Error = CRYS_DH_X942_GetSecret_Concat_2_Data(
				TST_UserPubKey,
				TST_PrimeData,
				CRYS_DH_HASH_SHA1_mode,
				TST_ClientPrvKey1,
				TST_ClientPrvKeySize1,
				TST_ServerPubKey1,
				TST_ServerPubKeySize1,
				Prime_ptr,
				PrimeSize,
				TST_ClientSecretKey,
				TST_ClientSecretKeySize); 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_GetSecret_Concat_2_Data - Client",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 


			/* Generating Secret key for the Server */
			/****************************************/                  
			TST_ATP_Error = CRYS_DH_X942_GetSecret_Concat_2_Data(
				TST_UserPubKey,
				TST_PrimeData,
				CRYS_DH_HASH_SHA1_mode,
				TST_ServerPrvKey1,
				TST_ServerPrvKeySize1,
				TST_ClientPubKey1,
				TST_ClientPubKeySize1,
				Prime_ptr,
				PrimeSize,
				TST_ServerSecretKey,
				TST_ServerSecretKeySize); 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_GetSecret_Concat_2_Data - Server",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 
		}
	}
	else
	{
		if (KeyGetSecretKeyType == TST_DH_GET_SECRET_KEY_ASN1)
		{
			/* Generating Secret key for the Client */
			/****************************************/                      
			TST_ATP_Error = CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data(
				TST_UserPubKey,
				TST_PrimeData,
				CRYS_DH_HASH_SHA1_mode,
				TST_ClientPrvKey1,
				TST_ClientPrvKeySize1,
				TST_ClientPrvKey2,
				TST_ClientPrvKeySize2,
				TST_ServerPubKey1,
				TST_ServerPubKeySize1,
				TST_ServerPubKey2,
				TST_ServerPubKeySize2,
				Prime_ptr,
				PrimeSize,
				TST_ClientSecretKey,
				TST_ClientSecretKeySize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data - Client", 
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 


			/* Generating Secret key for the Server */
			/****************************************/                      
			TST_ATP_Error = CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data(
				TST_UserPubKey,
				TST_PrimeData,
				CRYS_DH_HASH_SHA1_mode,
				TST_ServerPrvKey1,
				TST_ServerPrvKeySize1,
				TST_ServerPrvKey2,
				TST_ServerPrvKeySize2,
				TST_ClientPubKey1,
				TST_ClientPubKeySize1,
				TST_ClientPubKey2,
				TST_ClientPubKeySize2,
				Prime_ptr,
				PrimeSize,
				TST_ServerSecretKey,
				TST_ServerSecretKeySize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data - Server", 
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 
		}
		else
		{
			/* Generating Secret key for the Client */
			/****************************************/                      
			TST_ATP_Error = CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data(
				TST_UserPubKey,
				TST_PrimeData,
				CRYS_DH_HASH_SHA1_mode,
				TST_ClientPrvKey1,
				TST_ClientPrvKeySize1,
				TST_ClientPrvKey2,
				TST_ClientPrvKeySize2,
				TST_ServerPubKey1,
				TST_ServerPubKeySize1,
				TST_ServerPubKey2,
				TST_ServerPubKeySize2,
				Prime_ptr,
				PrimeSize,
				TST_ClientSecretKey,
				TST_ClientSecretKeySize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data - Client",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 

			/* Generating Secret key for the Server */
			/****************************************/                      
			TST_ATP_Error = CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data(
				TST_UserPubKey,
				TST_PrimeData, 
				CRYS_DH_HASH_SHA1_mode, 
				TST_ServerPrvKey1,
				TST_ServerPrvKeySize1,
				TST_ServerPrvKey2,
				TST_ServerPrvKeySize2,
				TST_ClientPubKey1,
				TST_ClientPubKeySize1,
				TST_ClientPubKey2,
				TST_ClientPubKeySize2,
				Prime_ptr,
				PrimeSize,
				TST_ServerSecretKey,
				TST_ServerSecretKeySize);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
				TST_ATP_Error,
				"CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data - Server",
				FailureFunctionNameOut);

			if (TST_TestStatus != TESTPASS) 
				goto End; 
		}               
	}

	/* 6. Verifying we have the same Secret Key both for client & Server */
	/*********************************************************************/                                                    
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_ServerSecretKey ,
											   TST_ClientSecretKey,
											   (KeyLenInBits)/8);  

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(
		TST_ATP_Error,
		"DX_VOS_MemCmp", 
		FailureFunctionNameOut);

End:
	DX_VOS_MemFree(TST_TempBuff);
	DX_VOS_MemFree(TST_UserPubKey);
	DX_VOS_MemFree(TST_PrimeData);
	DX_VOS_MemFree(TST_ClientPrvKey1);
	DX_VOS_MemFree(TST_ClientPubKey1);
	DX_VOS_MemFree(TST_ClientPrvKey2);
	DX_VOS_MemFree(TST_ClientPubKey2);
	DX_VOS_MemFree(TST_ServerPrvKey1);
	DX_VOS_MemFree(TST_ServerPubKey1);
	DX_VOS_MemFree(TST_ServerPrvKey2);
	DX_VOS_MemFree(TST_ServerPubKey2);
	DX_VOS_MemFree(TST_ClientSecretKey);
	DX_VOS_MemFree(TST_ServerSecretKey);

	return TST_TestStatus;	
}










